#!/usr/bin/env python

"""
downloaddialog.py

Copyright (C) 2006 David Boddie

This file is part of PyPI Browser, a GUI browser for the Python Package Index.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
"""

import os, urllib2, urlparse
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from ui_downloaddialog import Ui_DownloadDialog
from delegates import ProgressDelegate
import desktop


class DownloadDialog(QDialog, Ui_DownloadDialog):

    """DownloadDialog(QDialog, Ui_DownloadDialog)
    
    Provides a dialog that shows the progress of a series of download
    operations.
    
    Each package file that is successfully (or partially) downloaded is
    stored in a directory defined in the settings object specified when
    the dialog is created.
    """
    
    def __init__(self, settings, parent = None):
    
        QDialog.__init__(self, parent)
        self.setupUi(self)
        QMetaObject.connectSlotsByName(self)
        
        self.settings = settings
        self.stopped = False
        
        self.treeWidget.setColumnCount(2)
        self.treeWidget.setHeaderLabels(
            QStringList() << self.tr("Name") << self.tr("Progress")
            )
        delegate = ProgressDelegate(self)
        self.treeWidget.setItemDelegate(delegate)
        self.treeWidget.setMouseTracking(True)
        self.treeWidget.mouseMoveEvent = self._mouseMoveEvent
        
        font = QFont()
        font.setUnderline(True)
        self.linkFont = QVariant(font)
        
        self.connect(self.stopButton, SIGNAL("clicked()"), self.stopDownload)
        self.connect(self.treeWidget,
            SIGNAL("itemEntered(QTreeWidgetItem *, int)"),
            self.changeCursor)
        self.connect(self.treeWidget,
            SIGNAL("itemClicked(QTreeWidgetItem *, int)"),
            self.launchBrowser)
    
    def _mouseMoveEvent(self, event):
    
        item = self.treeWidget.itemAt(event.pos())
        if not item:
            self.treeWidget.unsetCursor()
        QTreeWidget.mouseMoveEvent(self.treeWidget, event)
    
    def changeCursor(self, item, column):
    
        if column == 1 and item.data(column, Qt.UserRole+1).isValid():
            self.treeWidget.setCursor(Qt.PointingHandCursor)
        else:
            self.treeWidget.unsetCursor()
    
    def downloadPackage(self, item, directory, download_url, completed, packages):
    
        try:
            u = urllib2.urlopen(download_url)
            info = u.info()
            length = int(info.getheader("Content-length"))
            
            path = urllib2.urlparse.urlsplit(download_url)[2]
            filename = path.split("/")[-1]
            savePath = os.path.join(directory, filename)
            
            f = open(savePath, "wb")
            
            total = 0
            while True:

                bytes = u.read(4096)
                read = len(bytes)
                total += read
                f.write(bytes)
                
                item.setText(1, self.tr("%1/%2 bytes (%3%)").arg(total).arg(length).arg(int(100*float(total)/length)))
                item.setData(1, Qt.UserRole, QVariant(100*float(total)/length))
                self.progressBar.setValue(100*(float(completed) + float(total)/length)/packages)
                qApp.processEvents()
                
                if read < 4096 or self.stopped:
                    break

            u.close()
            f.close()

        except:
            return False
        
        return True
    
    def execute(self, packages):
    
        """execute(self, packages)
        
        Download each of the packages in a list to a directory specified
        in the application's settings.
        
        If no suitable download directory is defined in the settings, the
        method returns immediately.
        
        The event loop is run periodically, enabling the progress of the
        download operation to be reporting and allowing the user to cancel
        the operation if required.
        """
        
        if not self.settings.value("Download directory").isValid():
            return
        
        directory = unicode(self.settings.value("Download directory").toString())
        if not os.path.isdir(directory):
            return
        
        self.startDownload()
        qApp.processEvents()
        
        completed = 0
        for name, version, download_url, home_url in packages:
        
            item = QTreeWidgetItem(self.treeWidget)
            item.setText(0, name)
            item.setText(1, self.tr("Fetching..."))
            item.setData(1, Qt.UserRole, QVariant(0))
            item.setFlags(Qt.ItemIsEnabled)
            
            downloaded = False
            if not download_url:
                home_url = download_url
            else:
                pieces = urlparse.urlsplit(download_url)
                path = pieces[2]
                if not path or path.endswith(u".html") or path.endswith(u"/"):
                    home_url = download_url
                elif u"." not in path.split(u"/")[-1]:
                    home_url = download_url
                else:
                    downloaded = self.downloadPackage(item, directory,
                                 download_url, completed, len(packages))
            
            if not downloaded:
            
                if not home_url or not urlparse.urlsplit(home_url)[0]:
                    item.setText(1, self.tr("Failed"))
                else:
                    item.setText(1, home_url)
                    item.setData(1, Qt.UserRole, QVariant())
                    item.setData(1, Qt.UserRole+1, QVariant(home_url))
                    item.setData(1, Qt.FontRole, self.linkFont)
            
            completed += 1
            
            self.progressBar.setValue(100*float(completed)/len(packages))
            qApp.processEvents()
            
            if self.stopped:
                break
        
        self.stopDownload()
    
    def launchBrowser(self, item, column):
    
        if column == 1:
            variant = item.data(column, Qt.UserRole+1)
            if variant.isValid():
                home_url = unicode(variant.toString())
                desktop.open(home_url)
    
    def startDownload(self):
    
        """startDownload(self)
        
        Prepares the user interface for use during a download operation.
        """
        
        self.stopped = False
        self.stopButton.setEnabled(True)
        self.closeButton.setEnabled(False)
        self.treeWidget.clear()
    
    def stopDownload(self):
    
        """stopDownload(self)
        
        Resets the user interface after a download operation.
        """
        
        self.stopped = True
        self.stopButton.setEnabled(False)
        self.closeButton.setEnabled(True)
    
    def reject(self):
    
        """reject(self)
        
        Stops any current download operation and rejects the dialog in the
        standard way.
        """
        
        self.stopDownload()
        QDialog.reject(self)
